# Binary Disk Graph Generator
# by Goetz Schwandtner <goetz@schwandtner.info> 2023-02-05
# researching the "Binary Disk" puzzle by Oskar van Deventer (c) 1988,2022

# V2 2023-03-04 Index shift fixed in processVertex
# V3 2023-03-11 adding colour highlight and switch for short dead end transitions, no double transitions yet!
# V4 2023-03-11 adding double transitions
# V5 2023-03-27 adding tooltips and removing labels, various parameter tweaks in GraphVIZ for different layout
# V6 2023-04-07 added some statistics based on shortest paths; they are put into comments at the output bottom
# V7 2023-05-01 adding pairwise calculations of shortest paths between goal configurations and roundtrip
# V8 2023-05-06 adding output of instructions for moves on shortest roundtrip, fixed roundtrip length

#### use of script:
# supply control values for red and blue as outlined below if required. Both will default to 0.
# stats=1 will enable output of statistics into comments in to file at the bottom of the output (default 0)
# stats=2 will also output the full distance list from the BFS, but not looking nice
# roundtrip=1 will compute the length of the shortest round trip of all 8 goal configurations
# roundtrip=2 will also add yellow arrows to the graph
# roundtrip=3 will also print the roundtrip into the file
#
# binaryDisk.py [redCtrlVal [blueCtrlVal [stats [roundtrip]]]]
#
#### examples:
# draw only graph without small groove use: binaryDisk.py
# draw graph with small grooves single transitions, but no colour highlight: binaryDisk.py 1 1
# draw graph with small grooves single transitions, and highlight them: binaryDisk.py 3 3
# draw graph with single and double transitions, and all highlights: binaryDisk.py 15 15

#### Definition bitmask for red and blue transition handling
# bit 0: 1 = with single transition for small groove
# bit 1: 1 = with colour highlight for single transition
# bit 2: 1 = with double transitions for small groove
# bit 3: 1 = with colour highlight for double transition small groove
# bit 4: 1 = with edge highlight for double transition small groove

#### examples:
# no use of red/blue small gaps: 0 (= 0_10)
# use only single transitions and highlight in colour: 3 (= 3_10 = 11_2)
# use all highlights for single and double transitions: 31 (= 31_10 = 11111_2)

import sys

from itertools import permutations

COLORS="RVPBGLYO"
#               0       1         2        3        4       5        6          7
COLORS_LONG= ["Red", "Violet", "Purple", "Blue", "Green", "Lime", "Yellow", "Orange"]

# format of transitions: TRANSITION[i] contains tuples (t, d) to move token t and new disk position
# transitions without red/blue in short ends
TRANSITIONS= [
    [(2,7),(3,1)], #R
    [(6,0),(7,2)], #V
    [(4,1),(5,3)], #P
    [(0,2),(1,4)], #B
    [(6,3),(7,5)], #G
    [(2,4),(3,6)], #L
    [(0,5),(1,7)], #Y
    [(4,6),(5,0)] #O
]

# double transitions: moves of red or blue token in small gap simultaneously with another one
# as usual, on transitions for red / blue : 0 |-> 1 are considered (the others are there by symmetry)

# format: list of other token ID 0..7 and starting position 0 (= marked) or 1 (=other)
REDDBLTRANS= [(1,1),(4,0)]
BLUEDBLTRANS= [(1,0),(6,1)]

vertexcnt= 0

# control value for red
redctrl= 0
# control value for blue
bluectrl= 0
# control value for statistics
stats= 0
# control value for roundtrip computation
roundtrip= 0

# variables for calculating shortest paths to solution configurations
# maybe not the cleanest solution with all these global variables and "global" keyword appearances ...
# adjacency list of graph. each entry is named by vertex name and contains list of adjacent names
adjacency= {}
# distances for each vertex. 0 for solution vertex, -1 initialization others, otherwise: number of edges to solution v
dists= {}
# queue for BFS, starting at solution vertices and then searching backwards
vqueue= []

# create a new vertex if required
def addvertex(name):
    if not name in adjacency:
        adjacency[name]= []
        dists[name]= -1

# add two edges: (v1, v2) and (v2, v1). Will initialize both vertices if not already present
def addedge(v1, v2):
    global adjacency
    addvertex(v1)
    addvertex(v2)
    adjacency[v1].append(v2)
    adjacency[v2].append(v1)

# create a nice string of a single configuration
def conf2string(conf):
    s=COLORS[conf[0]]+"_"
    for i in range(1,len(conf)):
        if conf[i] == 0:
            s += COLORS[i-1]
        else:
            s += COLORS[i-1].lower()
    return s

# create a transition (undirected edge in Dot) from src to tgt
def tr2str(src,tgt):
    # also add this to the local graph for calculation
    addedge(conf2string(src), conf2string(tgt))
    return conf2string(src)+" -- "+conf2string(tgt)

# create putputstring based on respective control setting, dbledge this being a double egde(true/false), and colour col
def getColhl(ctrl, dbledge, col):
    c= []
    if dbledge:
        if ctrl & 8 == 8:
            c.append("color = " + col)
        if ctrl & 16 == 16:
            c.append("penwidth=6")
            c.append("style=dashed")
    else:
        if ctrl & 2 == 2:
            c.append("color = "+col)
            c.append("penwidth=3")
    if len(c) == 0:
        return ""
    else:
        return "["+ ";".join(c) + "]"

def handleRedBlueSmall(conf):
    global redctrl, bluectrl
    # in orange state, the small gap for Red can be used (for Red in position 0)
    if conf[0] == 7:
        # single red transition
        if redctrl & 1 == 1:
            # check if red (digit 1 in conf) is in position 0, then transition it to 1
            if conf[1] == 0:
                conf2 = conf[:]
                conf2[0] = 6
                conf2[1] = 1
                print(tr2str(conf, conf2) +getColhl(redctrl,False,"red")+";")
        # double red transition, but only for red in position 0
        if redctrl & 4 == 4 and conf[1] == 0:
            for t,d in REDDBLTRANS:
                # check if other token is in appropriate state
                if conf[t+1] == d:
                    # create new conf with disk rotated, red set to 1, and token t flipped
                    conf2 = conf[:]
                    conf2[0] = 6
                    conf2[1] = 1
                    conf2[t+1] = 1-d
                    print(tr2str(conf, conf2) + getColhl(redctrl, True, "orangered") + ";")
    # in blue state, the small gap for Blue can be used (for Blue in position 0)
    elif conf[0] == 3:
        # single blue transition
        if bluectrl & 1 == 1:
            # check if blue (digit 4 in conf) is in position 0, then transition it to 1
            if conf[4] == 0:
                conf2 = conf[:]
                conf2[0] = 4
                conf2[4] = 1
                print(tr2str(conf, conf2) +getColhl(bluectrl,False,"blue")+";")
        # double red transition, but only for blue in position 0
        if bluectrl & 4 == 4 and conf[4] == 0:
            for t,d in BLUEDBLTRANS:
                # check if other token is in appropriate state
                if conf[t+1] == d:
                    # create new conf with disk rotated, red set to 1, and token t flipped
                    conf2 = conf[:]
                    conf2[0] = 4
                    conf2[4] = 1
                    conf2[t+1] = 1-d
                    print(tr2str(conf, conf2) + getColhl(bluectrl, True, "royalblue") + ";")

def processVertex(conf):
    global vertexcnt, dists, vqueue
    # different vertex output depending on whether it is a goal configuration or not
    tooltip="tooltip=\""+conf2string(conf)+"\" "
    if max(conf[1:]) == 0:  # special config, all tokens in start position
        print(conf2string(conf)+" [color=lime; fillcolor=lime; style=filled "+tooltip+"] ")
        # for computation, add this vertex if required and set his dist to 0
        addvertex(conf2string(conf))
        dists[conf2string(conf)]= 0
        # and also enqueue this vertex as one of the computation start vertices
        vqueue.append(conf2string(conf))
    else:
        print(conf2string(conf)+" [color=lightgrey; fillcolor=lightgrey; style=filled; "+tooltip+"] ")
    for (t, d) in TRANSITIONS[conf[0]]:
        if conf[t+1] == 0:
            conf2 = conf[:]
            conf2[0] = d
            conf2[t+1] = 1
            print(tr2str(conf, conf2)+";")
    handleRedBlueSmall(conf)
    vertexcnt+= 1

# main loop: hack to run through all configuration vertices and output
def traverseGraph():
    for dp in range(8):
        for r in range(2):
            for v in range(2):
                for p in range(2):
                    for b in range(2):
                        for g in range(2):
                            for l in range(2):
                                for y in range(2):
                                    for o in range (2):
                                        processVertex([dp,r,v,p,b,g,l,y,o])

# statistics computation: number of isolated vertices
def countIsolated():
    # our graph only contains vertices having at least one edge, hence subtract graph from all vertices ever touched
    return vertexcnt - len(adjacency)

# statistics computation: number of vertices not reachable from a solved state
def countNotReachable():
    notr= 0
    # count all vertices having dist of -1 after BFS
    for v in dists:
        if dists[v] == -1:
            notr += 1
    return notr

# statistics computation: maximum distance of a vertex to a solved state
def maxDistReachable():
    maxd= -1
    for v in dists:
        maxd = max(maxd, dists[v])
    return maxd

# computation: perform a BFS on the graph and set up the dists list
# ! this only works once and relies on the initialization above, including the vqueue
# output will be the dists list
def performBFS():
    global dists
    # as long as there are unprocessed but active vertices in the queue ...
    while vqueue:
        # get the first one of them from the queue
        v= vqueue.pop(0)
        # enqueue all of v's neighbors that have not been visited yet (dist of -1)
        for a in adjacency[v]:
            if dists[a] == -1:
                # new vertex is one step further away than v, so set distance accordingly
                dists[a]= dists[v] + 1
                vqueue.append(a)

# computation of statistics: run all
def outputStats():
    if stats == 2:
        print(f"# BFS distance list: {dists}")
    print(f"# Count of vertices total: {vertexcnt}")
    print(f"# Count of isolated vertices: {countIsolated()}")
    print(f"# Count of not reachable from solution: {countNotReachable()}")
    print(f"# Maximum distance from vertext to a solution vertex: {maxDistReachable()}")

# computation of pairwise shortest paths
# length of the paths
pwLens= [ [0]*8 for i in range(8) ]
# the actual paths (each one a list of vertex names)
pwPaths= [ [ [] for j in range(8) ] for i in range(8)]

# compute shortest path between one pair of end configurations
# start and end are given as index from 0..7 each
# this function adds the result both to path "istart -> iend" and "iend -> istart" in reverse
def pairwiseBFS(istart, iend):
    global pwLens, pwPaths
    if istart == iend:
        return
    # create names for vertices from these
    vstart= conf2string([istart, 0, 0, 0, 0, 0, 0, 0, 0])
    vend = conf2string([iend, 0, 0, 0, 0, 0, 0, 0, 0])
    # distance of each vertex from the start
    ldists= {}
    # queue of vertices to be still computed
    lqueue= []
    # predecessors of each vertex
    lpred= {}
    # start vertex entered
    ldists[vstart]= 0
    lqueue.append(vstart)
    while lqueue:
        # get the first one of them from the queue
        v= lqueue.pop(0)
        # enqueue all of v's neighbors that have not been visited yet (dist of -1)
        for a in adjacency[v]:
            # only vertices not yet handled
            if not a in ldists:
                # new vertex is one step further away than v, so set distance accordingly
                ldists[a]= ldists[v] + 1
                lqueue.append(a)
                # record the predecessor for path generation (a was reached from v)
                lpred[a]= v
                # should this be the end vertex already? Then clear the queue
                if a == vend:
                    lqueue= []
    # now compute the path by recording all the vertices
    lpath= [vend]
    v= vend
    while v != vstart:
        v= lpred[v]
        lpath.insert(0,v)
    pwPaths[istart][iend]= lpath
    lrev= lpath[:]
    lrev.reverse()
    pwPaths[iend][istart]= lrev
    pwLens[istart][iend]= len(lpath)-1
    pwLens[iend][istart]= len(lpath)-1

# compute all pairwise goal configuration paths and lengths
def pwPathsComputeAll():
    for s in range(7):
        for e in range(s+1, 8):
            pairwiseBFS(s, e)

rtminCycle= []
rtminLen= sys.maxsize

rtMoveDescription= []

# compute shortest roundtrip over all goal configurations
# as it always includes all of the 8 goal configurations, we just start at goal 7 (i.e. [7,0,0,0,0,0,0,0,0]) each time
def computeRoundtrip():
    global rtminCycle
    global rtminLen
    # get permutations of all but the first vertex 0
    permIter= permutations(range(1,8))
    for perm in permIter:
        path= [i for i in perm]
        # add the closing edge of the cycle first
        curLen= pwLens[path[6]][0]
        vst= 0
        for ven in path:
           curLen += pwLens[vst][ven]
           vst= ven
        # have we found a new minimum?
        if curLen < rtminLen:
            rtminLen= curLen
            rtminCycle= [0] + path

def drawRoundtripPath(start, end):
    global rtMoveDescription
    p=pwPaths[start][end]
    for i in range(len(p)-1):
        print(f"{p[i]} -- {p[i+1]}  [color=lime, penwidth=3]")
        rtMoveDescription.append(strInstMove(p[i],p[i+1]))

def drawRoundtrip():
    global rtMoveDescription
    l= len(rtminCycle)
    for p in range(l):
        rtMoveDescription.append(f"# Goal config (all tokens next to mark), disk pos: {COLORS_LONG[rtminCycle[p]]}")
        drawRoundtripPath(rtminCycle[p], rtminCycle[(p+1) % l])
    rtMoveDescription.append(f"# Goal config (all tokens next to mark), disk pos: {COLORS_LONG[rtminCycle[0]]}")

# create insstruction move description for round trip printout
def strInstMove(start, end):
    sstart = set(start[2:])
    send = set(end[2:])
    mv = send - sstart
    str = ""
    for m in mv:
        str += COLORS_LONG[COLORS.find(m.upper())] + " "
    return "# Move: "+str

def outputRoundtripLen():
    print(f"# Length of the shortest round trip through all goal vertices: {rtminLen}")

# functions to print move instructions for a path and then the whole round trip

def printRoundTrip():
    print("# #### Round trip move instructions:")
    for l in rtMoveDescription:
        print(l)

# main program including argument parser

def parseArgs():
    global redctrl, bluectrl, stats, roundtrip
    if len(sys.argv) > 1:
        redctrl = int(sys.argv[1])
    if len(sys.argv) > 2:
        bluectrl = int(sys.argv[2])
    if len(sys.argv) > 3:
        stats = int(sys.argv[3])
    if len(sys.argv) > 4:
        roundtrip = int(sys.argv[4])
        if roundtrip > 0 and ((redctrl & 4) | (bluectrl & 4) == 0):
            print("!!! Roundtrip only works with at least one gap available for double moves! Exiting ...")
            sys.exit(1)

if __name__ == '__main__':
    parseArgs()
    print("graph {")
    print("layout=neato; mode=major; model=mds; maxiter=500; overlap=compress;")
    print("node[label=\"\"; shape=circle; style=filled; height=0.2; width=0.2] ")
    print(f"comment=\"Parameter values: redctrl={redctrl} bluectrl={bluectrl}\"")
    traverseGraph()
    if stats > 0:
        performBFS()
        if roundtrip > 0:
            pwPathsComputeAll()
            computeRoundtrip()
        if roundtrip > 1:
            drawRoundtrip()
        outputStats()
        if roundtrip > 0:
            outputRoundtripLen()
        if roundtrip > 2:
            printRoundTrip()
    print("}")